<?php
/**
 * Helper Functions - v7.4.4
 */

function generateRandomKey($length = 12) {
    $chars = 'ABCDEFGHJKLMNPQRSTUVWXYZ23456789';
    $key = '';
    for ($i = 0; $i < $length; $i++) {
        $key .= $chars[random_int(0, strlen($chars) - 1)];
    }
    return $key;
}

function generateSessionToken() {
    return bin2hex(random_bytes(32));
}

function jsonResponse($data, $statusCode = 200) {
    http_response_code($statusCode);
    header('Content-Type: application/json');
    header('Access-Control-Allow-Origin: *');
    header('Access-Control-Allow-Methods: GET, POST, OPTIONS');
    header('Access-Control-Allow-Headers: Content-Type, X-Machine-ID, X-Device-ID, X-Session-Token');
    echo json_encode($data);
    exit;
}

function errorResponse($message, $statusCode = 400) {
    jsonResponse(['success' => false, 'error' => $message], $statusCode);
}

function successResponse($data = [], $message = 'Success') {
    jsonResponse(array_merge(['success' => true, 'message' => $message], $data));
}

function getJsonBody() {
    return json_decode(file_get_contents('php://input'), true) ?: [];
}

function getHeader($name) {
    $key = 'HTTP_' . strtoupper(str_replace('-', '_', $name));
    return $_SERVER[$key] ?? null;
}

function getClientIP() {
    $headers = ['HTTP_CF_CONNECTING_IP', 'HTTP_X_FORWARDED_FOR', 'HTTP_X_REAL_IP', 'REMOTE_ADDR'];
    foreach ($headers as $header) {
        if (!empty($_SERVER[$header])) {
            return explode(',', $_SERVER[$header])[0];
        }
    }
    return '0.0.0.0';
}

function validateSession($db, $machineId, $sessionToken) {
    // Try to find session by machine_id (with or without device_id)
    $sql = "SELECT s.*, u.is_active as user_active, u.id as user_id 
            FROM sessions s 
            JOIN users u ON s.user_id = u.id 
            WHERE u.machine_id = ? AND s.session_token = ? AND s.is_valid = 1 AND s.expires_at > NOW()";
    $session = $db->fetch($sql, [$machineId, $sessionToken]);
    
    if (!$session || !$session['user_active']) {
        return null;
    }
    
    $db->query("UPDATE users SET last_seen = NOW() WHERE id = ?", [$session['user_id']]);
    return $session;
}

function logActivity($db, $userId, $sessionId, $actionType, $data = []) {
    $db->insert('activity_logs', [
        'user_id' => $userId,
        'session_id' => $sessionId,
        'action_type' => $actionType,
        'company_name' => $data['company_name'] ?? null,
        'website_url' => $data['website_url'] ?? null,
        'status' => $data['status'] ?? null,
        'redirect_url' => $data['redirect_url'] ?? null,
        'filled_fields' => $data['filled_fields'] ?? null
    ]);
    
    updateDailyStats($db, $userId, $actionType);
}

function updateDailyStats($db, $userId, $actionType) {
    $today = date('Y-m-d');
    $existing = $db->fetch("SELECT id FROM daily_stats WHERE user_id = ? AND stat_date = ?", [$userId, $today]);
    
    $columnMap = [
        'success' => 'success_count',
        'skip' => 'skip_count', 
        'stay' => 'stay_count',
        'no_form' => 'no_form_count',
        'error' => 'error_count',
        'captcha' => 'captcha_count',
        'custom' => 'custom_count',
        'tab_closed' => 'tab_closed_count',
        'tab_opened' => 'tab_opened_count'
    ];
    
    $column = $columnMap[$actionType] ?? null;
    
    if ($existing) {
        if ($column) {
            $db->query("UPDATE daily_stats SET {$column} = {$column} + 1, total_processed = total_processed + 1 WHERE id = ?", [$existing['id']]);
        }
    } else {
        $data = ['user_id' => $userId, 'stat_date' => $today, 'total_processed' => 1];
        if ($column) $data[$column] = 1;
        $db->insert('daily_stats', $data);
    }
}

function handleCORS() {
    header('Access-Control-Allow-Origin: *');
    header('Access-Control-Allow-Methods: GET, POST, OPTIONS');
    header('Access-Control-Allow-Headers: Content-Type, X-Machine-ID, X-Device-ID, X-Session-Token');
    if ($_SERVER['REQUEST_METHOD'] === 'OPTIONS') {
        http_response_code(204);
        exit;
    }
}
