<?php
/**
 * Contact Form Pro - API Endpoint v7.4.4.2
 */

require_once __DIR__ . '/../includes/database.php';
require_once __DIR__ . '/../includes/helpers.php';

handleCORS();

$db = db();
$method = $_SERVER['REQUEST_METHOD'];
$action = $_GET['action'] ?? '';

// Helper function to check concurrent login
function checkConcurrentLogin($db, $userId, $machineId, $currentIp, $sessionId = null) {
    $user = $db->fetch("SELECT allow_concurrent, max_concurrent FROM users WHERE id = ?", [$userId]);
    if (!$user) return ['allowed' => false, 'reason' => 'User not found'];
    
    // Get active sessions from different IPs (last 5 minutes)
    $activeSessions = $db->fetchAll(
        "SELECT id, ip_address, created_at FROM sessions 
         WHERE user_id = ? AND is_valid = 1 AND ip_address != ? 
         AND created_at > DATE_SUB(NOW(), INTERVAL 5 MINUTE)
         ORDER BY created_at DESC",
        [$userId, $currentIp]
    );
    
    $concurrentCount = count($activeSessions) + 1;
    
    if ($concurrentCount > 1) {
        $allowConcurrent = isset($user['allow_concurrent']) ? $user['allow_concurrent'] : false;
        $maxConcurrent = isset($user['max_concurrent']) ? $user['max_concurrent'] : 2;
        
        if (!$allowConcurrent) {
            // Log blocked attempt
            try {
                $db->insert('concurrent_login_logs', [
                    'user_id' => $userId,
                    'machine_id' => $machineId,
                    'original_ip' => $activeSessions[0]['ip_address'] ?? 'unknown',
                    'blocked_ip' => $currentIp,
                    'original_session_id' => $activeSessions[0]['id'] ?? null,
                    'user_agent' => $_SERVER['HTTP_USER_AGENT'] ?? '',
                    'was_allowed' => 0
                ]);
            } catch (Exception $e) {}
            
            return [
                'allowed' => false, 
                'reason' => 'Concurrent login from different IP detected. Contact admin.'
            ];
        } else if ($concurrentCount > $maxConcurrent) {
            try {
                $db->insert('concurrent_login_logs', [
                    'user_id' => $userId,
                    'machine_id' => $machineId,
                    'original_ip' => $activeSessions[0]['ip_address'] ?? 'unknown',
                    'blocked_ip' => $currentIp,
                    'original_session_id' => $activeSessions[0]['id'] ?? null,
                    'user_agent' => $_SERVER['HTTP_USER_AGENT'] ?? '',
                    'was_allowed' => 0
                ]);
            } catch (Exception $e) {}
            
            return [
                'allowed' => false,
                'reason' => "Max concurrent logins ($maxConcurrent) exceeded."
            ];
        } else {
            // Allowed - log it
            try {
                $db->insert('concurrent_login_logs', [
                    'user_id' => $userId,
                    'machine_id' => $machineId,
                    'original_ip' => $activeSessions[0]['ip_address'] ?? 'unknown',
                    'blocked_ip' => $currentIp,
                    'original_session_id' => $activeSessions[0]['id'] ?? null,
                    'user_agent' => $_SERVER['HTTP_USER_AGENT'] ?? '',
                    'was_allowed' => 1
                ]);
            } catch (Exception $e) {}
        }
    }
    
    return ['allowed' => true];
}

try {
    switch ($action) {
        
        case 'register':
            if ($method !== 'POST') errorResponse('Method not allowed', 405);
            
            $data = getJsonBody();
            $machineId = $data['machine_id'] ?? '';
            $deviceId = $data['device_id'] ?? getHeader('X-Device-ID') ?? '';
            
            if (empty($machineId) || strlen($machineId) < 10) {
                errorResponse('Invalid machine ID');
            }
            
            // Check for existing user by machine_id AND device_id
            $existing = $db->fetch(
                "SELECT * FROM users WHERE machine_id = ? AND (device_id = ? OR device_id IS NULL)", 
                [$machineId, $deviceId]
            );
            
            if ($existing) {
                // Update device_id if not set
                if (empty($existing['device_id']) && !empty($deviceId)) {
                    $db->query("UPDATE users SET device_id = ? WHERE id = ?", [$deviceId, $existing['id']]);
                }
                successResponse([
                    'registered' => true,
                    'is_active' => (bool)$existing['is_active'],
                    'needs_key' => true
                ], 'Already registered');
            } else {
                // Check if this machine_id exists with different device_id (different device)
                $otherDevice = $db->fetch("SELECT * FROM users WHERE machine_id = ?", [$machineId]);
                if ($otherDevice && !empty($deviceId) && $otherDevice['device_id'] !== $deviceId) {
                    // This is a different device with same fingerprint - create new user
                    $loginKey = generateRandomKey(12);
                    $userId = $db->insert('users', [
                        'machine_id' => $machineId,
                        'device_id' => $deviceId,
                        'login_key' => $loginKey,
                        'last_ip' => getClientIP()
                    ]);
                    
                    successResponse([
                        'registered' => true,
                        'user_id' => $userId,
                        'needs_key' => true,
                        'new_device' => true
                    ], 'New device registered. Contact admin for login key.');
                } else {
                    // Brand new registration
                    $loginKey = generateRandomKey(12);
                    $userId = $db->insert('users', [
                        'machine_id' => $machineId,
                        'device_id' => $deviceId,
                        'login_key' => $loginKey,
                        'last_ip' => getClientIP()
                    ]);
                    
                    successResponse([
                        'registered' => true,
                        'user_id' => $userId,
                        'needs_key' => true
                    ], 'Registered. Contact admin for login key.');
                }
            }
            break;
            
        case 'login':
            if ($method !== 'POST') errorResponse('Method not allowed', 405);
            
            $data = getJsonBody();
            $machineId = $data['machine_id'] ?? '';
            $deviceId = $data['device_id'] ?? getHeader('X-Device-ID') ?? '';
            $loginKey = strtoupper(trim($data['login_key'] ?? ''));
            
            if (empty($machineId) || empty($loginKey)) {
                errorResponse('Machine ID and Login Key required');
            }
            
            // Find user by machine_id + login_key, optionally with device_id
            $user = null;
            if (!empty($deviceId)) {
                // Try to match with device_id first
                $user = $db->fetch(
                    "SELECT * FROM users WHERE machine_id = ? AND device_id = ? AND login_key = ?", 
                    [$machineId, $deviceId, $loginKey]
                );
            }
            
            if (!$user) {
                // Fallback to just machine_id + login_key
                $user = $db->fetch(
                    "SELECT * FROM users WHERE machine_id = ? AND login_key = ?", 
                    [$machineId, $loginKey]
                );
                
                // If found and device_id is empty, update it
                if ($user && empty($user['device_id']) && !empty($deviceId)) {
                    $db->query("UPDATE users SET device_id = ? WHERE id = ?", [$deviceId, $user['id']]);
                } elseif ($user && !empty($user['device_id']) && !empty($deviceId) && $user['device_id'] !== $deviceId) {
                    // Different device trying to use same key - BLOCK
                    errorResponse('This login key is already registered to another device', 403);
                }
            }
            
            if (!$user) {
                errorResponse('Invalid login key', 401);
            }
            
            if (!$user['is_active']) {
                errorResponse('Account disabled', 403);
            }
            
            // Invalidate old sessions
            $db->query("UPDATE sessions SET is_valid = 0 WHERE user_id = ?", [$user['id']]);
            
            // Create new session
            $sessionToken = generateSessionToken();
            $expiresAt = date('Y-m-d H:i:s', time() + SESSION_DURATION);
            
            $sessionId = $db->insert('sessions', [
                'user_id' => $user['id'],
                'session_token' => $sessionToken,
                'ip_address' => getClientIP(),
                'user_agent' => $_SERVER['HTTP_USER_AGENT'] ?? '',
                'expires_at' => $expiresAt
            ]);
            
            $db->query("UPDATE users SET last_seen = NOW(), last_ip = ? WHERE id = ?", [getClientIP(), $user['id']]);
            
            successResponse([
                'session_token' => $sessionToken,
                'expires_at' => $expiresAt,
                'nickname' => $user['nickname']
            ], 'Login successful');
            break;
            
        case 'verify':
            if ($method !== 'POST' && $method !== 'GET') errorResponse('Method not allowed', 405);
            
            $machineId = getHeader('X-Machine-ID');
            $sessionToken = getHeader('X-Session-Token');
            
            if (empty($machineId) || empty($sessionToken)) {
                errorResponse('Headers required', 401);
            }
            
            $session = validateSession($db, $machineId, $sessionToken);
            if (!$session) {
                errorResponse('Invalid session', 401);
            }
            
            // Check for concurrent login
            $currentIp = getClientIP();
            $concurrentCheck = checkConcurrentLogin($db, $session['user_id'], $machineId, $currentIp, $session['id']);
            
            if (!$concurrentCheck['allowed']) {
                $db->query("UPDATE sessions SET is_valid = 0 WHERE id = ?", [$session['id']]);
                errorResponse($concurrentCheck['reason'], 403);
            }
            
            // Update IP if changed
            if (isset($session['ip_address']) && $session['ip_address'] !== $currentIp) {
                $db->query("UPDATE sessions SET ip_address = ? WHERE id = ?", [$currentIp, $session['id']]);
            }
            
            successResponse(['valid' => true]);
            break;
            
        case 'logout':
            $machineId = getHeader('X-Machine-ID');
            $sessionToken = getHeader('X-Session-Token');
            
            if ($machineId && $sessionToken) {
                $db->query(
                    "UPDATE sessions s JOIN users u ON s.user_id = u.id SET s.is_valid = 0 WHERE u.machine_id = ? AND s.session_token = ?",
                    [$machineId, $sessionToken]
                );
            }
            successResponse([], 'Logged out');
            break;
            
        case 'log':
            if ($method !== 'POST') errorResponse('Method not allowed', 405);
            
            $machineId = getHeader('X-Machine-ID');
            $sessionToken = getHeader('X-Session-Token');
            
            $session = validateSession($db, $machineId, $sessionToken);
            if (!$session) {
                errorResponse('Invalid session', 401);
            }
            
            $data = getJsonBody();
            $actionType = $data['action_type'] ?? '';
            
            $validActions = ['success', 'skip', 'stay', 'no_form', 'error', 'captcha', 'custom', 'tab_closed', 'tab_opened'];
            if (!in_array($actionType, $validActions)) {
                errorResponse('Invalid action type');
            }
            
            logActivity($db, $session['user_id'], $session['id'], $actionType, $data);
            successResponse([], 'Logged');
            break;
            
        case 'log_batch':
            if ($method !== 'POST') errorResponse('Method not allowed', 405);
            
            $machineId = getHeader('X-Machine-ID');
            $sessionToken = getHeader('X-Session-Token');
            
            $session = validateSession($db, $machineId, $sessionToken);
            if (!$session) {
                errorResponse('Invalid session', 401);
            }
            
            $data = getJsonBody();
            $activities = $data['activities'] ?? [];
            $logged = 0;
            
            foreach ($activities as $activity) {
                $actionType = $activity['action_type'] ?? '';
                if (in_array($actionType, ['success', 'skip', 'stay', 'no_form', 'error', 'captcha', 'custom', 'tab_closed', 'tab_opened'])) {
                    logActivity($db, $session['user_id'], $session['id'], $actionType, $activity);
                    $logged++;
                }
            }
            
            successResponse(['logged' => $logged]);
            break;
            
        case 'stats':
            $machineId = getHeader('X-Machine-ID');
            $sessionToken = getHeader('X-Session-Token');
            
            $session = validateSession($db, $machineId, $sessionToken);
            if (!$session) {
                errorResponse('Invalid session', 401);
            }
            
            $userId = $session['user_id'];
            
            $today = $db->fetch("SELECT * FROM daily_stats WHERE user_id = ? AND stat_date = CURDATE()", [$userId]) ?: [];
            
            $month = $db->fetch("
                SELECT COALESCE(SUM(success_count),0) as success, COALESCE(SUM(skip_count),0) as skip,
                       COALESCE(SUM(stay_count),0) as stay, COALESCE(SUM(total_processed),0) as total,
                       COALESCE(SUM(tab_opened_count),0) as tabs_opened
                FROM daily_stats WHERE user_id = ? AND MONTH(stat_date) = MONTH(CURDATE()) AND YEAR(stat_date) = YEAR(CURDATE())
            ", [$userId]);
            
            $total = $db->fetch("
                SELECT COALESCE(SUM(success_count),0) as success, COALESCE(SUM(skip_count),0) as skip,
                       COALESCE(SUM(stay_count),0) as stay, COALESCE(SUM(total_processed),0) as total,
                       COALESCE(SUM(tab_opened_count),0) as tabs_opened
                FROM daily_stats WHERE user_id = ?
            ", [$userId]);
            
            successResponse(['today' => $today, 'month' => $month, 'total' => $total]);
            break;
            
        case 'check_access':
            // Check if extension version is allowed to run
            // This allows you to force updates or disable the extension remotely
            $data = getJsonBody();
            $clientVersion = $data['version'] ?? '0.0.0';
            
            // Read settings from config or database
            // You can update these values on server to control extension
            $minVersion = defined('MIN_EXTENSION_VERSION') ? MIN_EXTENSION_VERSION : '7.0.0';
            $forceUpdate = defined('FORCE_EXTENSION_UPDATE') ? FORCE_EXTENSION_UPDATE : false;
            $extensionEnabled = defined('EXTENSION_ENABLED') ? EXTENSION_ENABLED : true;
            
            if (!$extensionEnabled) {
                errorResponse('Extension temporarily disabled for maintenance', 503);
            }
            
            successResponse([
                'allowed' => true,
                'min_version' => $minVersion,
                'force_update' => $forceUpdate,
                'message' => ''
            ]);
            break;
            
        case 'get_app':
            // Serve the main application code (HTML, CSS, JS) to authenticated users
            $machineId = getHeader('X-Machine-ID');
            $sessionToken = getHeader('X-Session-Token');
            
            $session = validateSession($db, $machineId, $sessionToken);
            if (!$session) {
                errorResponse('Invalid session', 401);
            }
            
            // Load app files from server
            $appDir = __DIR__ . '/../app/';
            
            $html = file_exists($appDir . 'popup-content.html') 
                ? file_get_contents($appDir . 'popup-content.html') 
                : '';
            $css = file_exists($appDir . 'popup-style.css') 
                ? file_get_contents($appDir . 'popup-style.css') 
                : '';
            $js = file_exists($appDir . 'popup-app.js') 
                ? file_get_contents($appDir . 'popup-app.js') 
                : '';
            
            if (empty($html) || empty($js)) {
                errorResponse('App files not found on server', 500);
            }
            
            // Get user info
            $user = $db->fetch("SELECT nickname FROM users WHERE id = ?", [$session['user_id']]);
            
            successResponse([
                'html' => $html,
                'css' => $css,
                'js' => $js,
                'nickname' => $user['nickname'] ?? '',
                'version' => '7.4.4.1'
            ]);
            break;
            
        default:
            errorResponse('Unknown action', 404);
    }
} catch (Exception $e) {
    error_log("API Error: " . $e->getMessage());
    errorResponse('Server error', 500);
}
